using System;
using System.Reflection;
using System.Data;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.ExceptionManagement;
using DONATIONTYPE = gov.va.med.vbecs.Common.VbecsTables.DonationType;

namespace gov.va.med.vbecs.BOL
{


		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/6/2002</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		///Donation Type business object layer class
		///</summary>

		#endregion


	public class DonationType: BaseBusinessObject
	{
		private char donationTypeCode = char.MinValue;
		private string donationTypeText = string.Empty;
		private string directedType = string.Empty;
		private int _donationTypeId = -1;

		#region Events

		/// <summary>
		/// TestingStatusChanged event handler
		/// </summary>
		public event EventHandler TestingStatusChanged;
		/// <summary>
		/// Directed Type changed event
		/// </summary>
		public event EventHandler DirectedTypeChanged;
		/// <summary>
		/// DonationTypeCode changed event
		/// </summary>
		public event EventHandler DonationTypeCodeChanged;

		/// <summary>
		/// Directed Type changed event
		/// </summary>
		protected virtual void OnDirectedTypeChanged()
		{
			if(DirectedTypeChanged != null)
				DirectedTypeChanged(this, EventArgs.Empty);
		}

		/// <summary>
		/// DonationType changed event
		/// </summary>
		protected virtual void OnDonationTypeCodeChanged()
		{
			if(DonationTypeCodeChanged != null)
				DonationTypeCodeChanged(this, EventArgs.Empty);
		}

		/// <summary>
		/// BiohazardousDonationType changed event
		/// </summary>
		protected virtual void OnTestingStatusChanged()
		{
			if(TestingStatusChanged != null)
				TestingStatusChanged(this, EventArgs.Empty);
		}
		#endregion


		#region Constructors

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="545"> 
		///		<ExpectedInput>Valid donation type code</ExpectedInput>
		///		<ExpectedOutput>Valid DonationType object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2294"> 
		///		<ExpectedInput>Invalid donation type code</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Overloaded Donation Type class constructor, pre-load values using given donation type code
		/// </summary>
		/// <param name="donationTypeCode"></param>
		public DonationType(char donationTypeCode)
		{
			DataRow drDonation;
			try
			{
				 drDonation = BOL.DonationType.GetDonationTypeByCode(donationTypeCode);
			}
			catch(Exception ex)
			{
				throw new BusinessObjectException(Common.StrRes.SysErrMsg.Common.GeneralError(
					Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString, ex);	
			}
			this.DonationTypeCode = donationTypeCode;
			this._donationTypeId = (int)drDonation[DONATIONTYPE.DonationTypeId];
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1123"> 
		///		<ExpectedInput>Valid donation type ID</ExpectedInput>
		///		<ExpectedOutput>Valid object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1124"> 
		///		<ExpectedInput>Invalid donation type ID</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructor.  Takes in donation type ID as parameter 
		/// </summary>
		/// <param name="donationTypeID"></param>
		public DonationType(int donationTypeID)
		{
			System.Data.DataRow dr = DAL.DonationType.GetDonationTypeByID(donationTypeID);

			this.LoadFromDataRow(dr);

		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2405"> 
		///		<ExpectedInput>DataRow containing DonationType data</ExpectedInput>
		///		<ExpectedOutput>DonationType object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="16"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates DonationType object and populates it with data from DataRow
		/// </summary>
		/// <param name="dr"></param>
		public DonationType(DataRow dr)
		{
			this.LoadFromDataRow(dr);
		}

		#endregion Constructors


		#region Properties





		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="548"> 
		///		<ExpectedInput>Valid donation type code</ExpectedInput>
		///		<ExpectedOutput>Valid donation type code</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2308"> 
		///		<ExpectedInput>InValid donation type code</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets donation type code
		/// </summary>
		public char DonationTypeCode
		{
			get
			{
				return this.donationTypeCode;
			}
			set
			{
				this.donationTypeCode = value;
				SetDirectedType(value);
				this._donationTypeId = Common.Utility.GetDonationTypeIdFromChar(value);
				if(value != char.MinValue && value != 'U' && value != '0' && value != 'Q')
				{
					System.Data.DataRow dtRow = BOL.DonationType.GetDonationTypeByCode(value);
					if(dtRow != null)
					{
						if(dtRow.Table.Columns.Contains(DONATIONTYPE.DonationTypeText))
						{
							this.donationTypeText = dtRow[DONATIONTYPE.DonationTypeText].ToString();
						}
					}
				}
				RuleBroken(MethodBase.GetCurrentMethod().Name,(value == 'U' || value == '0' || value == 'Q'));
				this.OnDonationTypeCodeChanged();
				this.OnTestingStatusChanged();
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6297"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6298"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SetDirectedType
		/// </summary>
		/// <param name="donationTypeCode">Donation Type Code</param>
		public void SetDirectedType(char donationTypeCode)
		{
			switch(donationTypeCode.ToString())
			{
					//All of these are autologous
				case "A":
					this.DirectedType = "A";
					break;
				case "1":
					this.DirectedType = "A";
					break;
				case "X":
					this.DirectedType = "A";
					break;
				case "R":
					this.DirectedType = "D";
					break;
				case "S":
					this.DirectedType = "D";
					break;
				case "T":
					this.DirectedType = "D";
					break;
				case "P":
					this.DirectedType = "D";
					break;
				case "r":
					this.DirectedType = "D";
					break;
				case "s":
					this.DirectedType = "D";
					break;
				case "D":
					this.DirectedType = "D";
					break;
				case "d":
					this.DirectedType = "D";
					break;
				case "2":
					this.DirectedType = "D";
					break;
				case "L":
					this.DirectedType = "D";
					break;
				case "E":
					this.DirectedType = "D";
					break;
				case "3":
					this.DirectedType = "D";
					break;
				case "4":
					this.DirectedType = "D";
					break;
				case "5":
					this.DirectedType = "D";
					break;
				default:
					this.DirectedType = "O";
					break;
			}
		}

		
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="549"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Donation type text</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2309"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets donation type description
		/// </summary>
		public string DonationTypeText
		{
			get
			{
				if((this.donationTypeText.Trim().Length == 0) && (this.donationTypeCode != char.MinValue))
				{
					System.Data.DataRow dtRow = DAL.DonationType.GetDonationTypeByCode(this.DonationTypeCode);
					this.donationTypeText = dtRow[DONATIONTYPE.DonationTypeText].ToString().Trim();
				}
				return this.donationTypeText;
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1126"> 
		///		<ExpectedInput>Allogeneic entry</ExpectedInput>
		///		<ExpectedOutput>No change in broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2310"> 
		///		<ExpectedInput>Test string e.g. Test</ExpectedInput>
		///		<ExpectedOutput>Broken rules should be removed</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Type of donation - Autologous, Directed, or Other e.g. Allogeneic)
		/// Implemented BR_1.35
		/// </summary>
		public string DirectedType
		{
			get
			{
				return this.directedType;
			}

			set
			{
				//BR_1.35
				this.directedType = value;
				switch (this.directedType)
				{
					case "A":
						RuleBroken("TestingStatus",true);
						RuleBroken("Patient",true);
						break;
					case "D":
						RuleBroken("TestingStatus",false);
						RuleBroken("Patient",true);
						break;
					default:
						RuleBroken("TestingStatus",false);
						RuleBroken("Patient",false);
						break;
				}
				this.OnDirectedTypeChanged();
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6299"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6300"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// DonationType's primary key
		/// </summary>
		public int DonationTypeId
		{
			get
			{
				return this._donationTypeId;
			}
		}

		#endregion


		#region Methods

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="550"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Data table of donation type lookup entries</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="17"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function to return a data table of donation types. Can be used to populate a combobox.
		/// </summary>
		/// <returns></returns>
		public static DataTable GetDonationTypeLookupDataTable()
		{
			return (DAL.DonationType.GetDonationTypeLookupList());
		}
		
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6301"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6302"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///		Retrieves an array list of donation types.
		/// </summary>
		/// <returns>Array List of donation types</returns>
		public static System.Collections.ArrayList GetDonationTypeLookupList()
		{
			System.Collections.ArrayList al;
			al = new System.Collections.ArrayList();
			try
			{				
				System.Data.DataTable dt = gov.va.med.vbecs.DAL.DonationType.GetDonationTypeLookupList();
				foreach(System.Data.DataRow dtRow in dt.Rows)
				{
					BOL.DonationType donationType = new BOL.DonationType(dtRow);
					al.Add(donationType);
				}
				return al;
			}
			catch(Common.DataAccessLayerException)
			{
				throw;
			}
			catch(Exception ex)
			{
				ExceptionManager.Publish(ex);
				throw new BusinessObjectException(Common.StrRes.SysErrMsg.Common.GeneralError(
					Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString, ex);	
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="561"> 
		///		<ExpectedInput>Valid donation type code</ExpectedInput>
		///		<ExpectedOutput>Valid data row</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2329"> 
		///		<ExpectedInput>Invalid donation type code</ExpectedInput>
		///		<ExpectedOutput>Invalid data row</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function to return a data row with details of supplied donation type code
		/// </summary>
		/// <param name="donationTypeCode">char</param>
		/// <returns></returns>
		public static DataRow GetDonationTypeByCode(char donationTypeCode)
		{
			DataRow _dr;
			//
			try
			{
				_dr = DAL.DonationType.GetDonationTypeByCode(donationTypeCode);
			}
			catch(Common.DataAccessLayerException)
			{
				throw;
			}
			catch(Exception ex)
			{
				ExceptionManager.Publish(ex);
				throw new BusinessObjectException(Common.StrRes.SysErrMsg.Common.GeneralError(
					Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString, ex);	
			}
			//
			return _dr;
		}
		
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6303"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6304"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetDonationTypeCodeById
		/// </summary>
		/// <param name="donationTypeId"></param>
		/// <returns></returns>
		public static char GetDonationTypeCodeById(int donationTypeId)
		{
			DataRow _dr;
			char donationTypeCode = char.MinValue;
			//
			try
			{
				_dr = DAL.DonationType.GetDonationTypeByID(donationTypeId);
			}
			catch(Common.DataAccessLayerException)
			{
				throw;
			}
			catch(Exception ex)
			{
				ExceptionManager.Publish(ex);
				throw new BusinessObjectException(Common.StrRes.SysErrMsg.Common.GeneralError(
					Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString, ex);	
			}
			//
			if(!_dr.IsNull("DonationTypeCode"))
			{
				donationTypeCode = System.Convert.ToChar(_dr[DONATIONTYPE.DonationTypeCode]);
			}
			return donationTypeCode;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1128"> 
		///		<ExpectedInput>Valid data row</ExpectedInput>
		///		<ExpectedOutput>Valid data row</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="18"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Loads a data row from an object
		/// </summary>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			return dtRow;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads an object from a data row
		/// </summary>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(DONATIONTYPE.DonationTypeCode))
			{
				if (dtRow.IsNull(DONATIONTYPE.DonationTypeCode))
					this.DonationTypeCode = Char.MinValue;
				else
					this.DonationTypeCode = System.Convert.ToChar(dtRow[DONATIONTYPE.DonationTypeCode]);
			}

			if (dtRow.Table.Columns.Contains(DONATIONTYPE.DonationTypeText))
			{
				if (dtRow.IsNull(DONATIONTYPE.DonationTypeText))
					this.donationTypeText = string.Empty;
				else
					this.donationTypeText = dtRow[DONATIONTYPE.DonationTypeText].ToString();
			}

			if(dtRow.Table.Columns.Contains(DONATIONTYPE.DonationTypeId))
			{
				if (dtRow.IsNull(DONATIONTYPE.DonationTypeId))
					this._donationTypeId = -1;
				else
					this._donationTypeId = (int)dtRow[DONATIONTYPE.DonationTypeId];
			}


			if(dtRow.Table.Columns.Contains(Common.DatabaseConstants.ArtificialColumnNames.DirectedType))
			{
				if (dtRow.IsNull(Common.DatabaseConstants.ArtificialColumnNames.DirectedType))
					this.DirectedType = string.Empty;
				else
					this.DirectedType = dtRow[Common.DatabaseConstants.ArtificialColumnNames.DirectedType].ToString();
			}

		}



		#endregion
	}
}


